/**
 * Apple-Style Mortgage Site JavaScript
 * Progressive enhancement with smooth animations
 */

(function() {
  'use strict';

  // Check for reduced motion preference
  const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  // Intersection Observer for fade-in animations
  class ScrollAnimations {
    constructor() {
      this.observerOptions = {
        root: null,
        rootMargin: '0px 0px -50px 0px',
        threshold: 0.1
      };
      
      this.init();
    }

    init() {
      if (prefersReducedMotion || !('IntersectionObserver' in window)) {
        // Make all elements visible immediately if reduced motion or no support
        document.querySelectorAll('[data-animate]').forEach(el => {
          el.style.opacity = '1';
          el.style.transform = 'translateY(0)';
        });
        return;
      }

      this.setupObserver();
      this.setupElements();
    }

    setupObserver() {
      this.observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            this.animateElement(entry.target);
            this.observer.unobserve(entry.target);
          }
        });
      }, this.observerOptions);
    }

    setupElements() {
      // Set initial states for animated elements
      document.querySelectorAll('[data-animate]').forEach(el => {
        const animationType = el.dataset.animate;
        
        switch (animationType) {
          case 'fade-in':
            el.style.opacity = '0';
            break;
          case 'slide-up':
            el.style.opacity = '0';
            el.style.transform = 'translateY(40px)';
            break;
          case 'slide-in-left':
            el.style.opacity = '0';
            el.style.transform = 'translateX(-40px)';
            break;
          case 'slide-in-right':
            el.style.opacity = '0';
            el.style.transform = 'translateX(40px)';
            break;
        }
        
        el.style.transition = 'all 0.6s cubic-bezier(0.22, 1, 0.36, 1)';
        this.observer.observe(el);
      });
    }

    animateElement(element) {
      const delay = element.dataset.delay || 0;
      
      setTimeout(() => {
        element.style.opacity = '1';
        element.style.transform = 'translateY(0) translateX(0)';
      }, parseInt(delay));
    }
  }

  // Parallax effect for hero image (subtle)
  class ParallaxEffect {
    constructor() {
      this.elements = document.querySelectorAll('[data-parallax]');
      this.init();
    }

    init() {
      if (prefersReducedMotion || this.elements.length === 0) return;

      // Only enable on desktop to avoid performance issues on mobile
      if (window.innerWidth > 1024) {
        this.bindEvents();
      }
    }

    bindEvents() {
      let ticking = false;

      const updateParallax = () => {
        this.updateElements();
        ticking = false;
      };

      const requestTick = () => {
        if (!ticking) {
          requestAnimationFrame(updateParallax);
          ticking = true;
        }
      };

      window.addEventListener('scroll', requestTick, { passive: true });
    }

    updateElements() {
      const scrolled = window.pageYOffset;
      const rate = scrolled * -0.3; // Subtle parallax

      this.elements.forEach(element => {
        element.style.transform = `translateY(${rate}px)`;
      });
    }
  }

  // Mobile Navigation
  class MobileNavigation {
    constructor() {
      this.burger = document.querySelector('[data-burger]');
      this.nav = document.querySelector('[data-mobile-nav]');
      this.overlay = document.querySelector('[data-nav-overlay]');
      this.isOpen = false;
      
      this.init();
    }

    init() {
      if (!this.burger || !this.nav) return;
      
      this.bindEvents();
      this.setupAccessibility();
    }

    bindEvents() {
      this.burger.addEventListener('click', () => this.toggle());
      
      if (this.overlay) {
        this.overlay.addEventListener('click', () => this.close());
      }

      // Close on escape key
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && this.isOpen) {
          this.close();
        }
      });

      // Close on window resize
      window.addEventListener('resize', () => {
        if (window.innerWidth > 1024 && this.isOpen) {
          this.close();
        }
      });
    }

    setupAccessibility() {
      this.burger.setAttribute('aria-expanded', 'false');
      this.burger.setAttribute('aria-controls', 'mobile-navigation');
      this.nav.setAttribute('id', 'mobile-navigation');
    }

    toggle() {
      this.isOpen ? this.close() : this.open();
    }

    open() {
      this.isOpen = true;
      this.burger.setAttribute('aria-expanded', 'true');
      this.nav.classList.add('is-open');
      if (this.overlay) this.overlay.classList.add('is-open');
      document.body.style.overflow = 'hidden';
    }

    close() {
      this.isOpen = false;
      this.burger.setAttribute('aria-expanded', 'false');
      this.nav.classList.remove('is-open');
      if (this.overlay) this.overlay.classList.remove('is-open');
      document.body.style.overflow = '';
    }
  }

  // Smooth scrolling for anchor links
  class SmoothScroll {
    constructor() {
      this.init();
    }

    init() {
      document.addEventListener('click', (e) => {
        const link = e.target.closest('a[href^="#"]');
        if (!link) return;

        const targetId = link.getAttribute('href').substring(1);
        const target = document.getElementById(targetId);
        
        if (!target) return;

        e.preventDefault();
        this.scrollToTarget(target);
      });
    }

    scrollToTarget(target) {
      const headerHeight = document.querySelector('.header')?.offsetHeight || 0;
      const targetPosition = target.offsetTop - headerHeight - 20;

      window.scrollTo({
        top: targetPosition,
        behavior: 'smooth'
      });
    }
  }

  // Form Enhancement
  class FormEnhancement {
    constructor() {
      this.forms = document.querySelectorAll('form[data-enhance]');
      this.init();
    }

    init() {
      this.forms.forEach(form => this.enhanceForm(form));
    }

    enhanceForm(form) {
      const inputs = form.querySelectorAll('input, textarea, select');
      
      inputs.forEach(input => {
        this.enhanceInput(input);
      });

      form.addEventListener('submit', (e) => this.handleSubmit(e, form));
    }

    enhanceInput(input) {
      const wrapper = input.closest('.form-field');
      if (!wrapper) return;

      // Float label effect
      const label = wrapper.querySelector('label');
      if (label) {
        input.addEventListener('focus', () => wrapper.classList.add('is-focused'));
        input.addEventListener('blur', () => {
          wrapper.classList.remove('is-focused');
          wrapper.classList.toggle('has-value', input.value !== '');
        });

        // Check initial value
        if (input.value !== '') {
          wrapper.classList.add('has-value');
        }
      }

      // Real-time validation
      input.addEventListener('input', () => this.validateInput(input));
    }

    validateInput(input) {
      const wrapper = input.closest('.form-field');
      if (!wrapper) return;

      const isValid = input.checkValidity();
      wrapper.classList.toggle('is-invalid', !isValid);
      wrapper.classList.toggle('is-valid', isValid && input.value !== '');
    }

    handleSubmit(e, form) {
      e.preventDefault();
      
      // Basic form validation
      const inputs = form.querySelectorAll('input[required], textarea[required], select[required]');
      let isFormValid = true;

      inputs.forEach(input => {
        if (!input.checkValidity()) {
          isFormValid = false;
          this.validateInput(input);
        }
      });

      if (isFormValid) {
        this.submitForm(form);
      } else {
        // Focus first invalid input
        const firstInvalid = form.querySelector('.is-invalid input, .is-invalid textarea, .is-invalid select');
        if (firstInvalid) firstInvalid.focus();
      }
    }

    submitForm(form) {
      const submitBtn = form.querySelector('button[type="submit"]');
      const originalText = submitBtn.textContent;
      
      // Show loading state
      submitBtn.textContent = 'Submitting...';
      submitBtn.disabled = true;

      // Simulate form submission (replace with actual endpoint)
      setTimeout(() => {
        submitBtn.textContent = 'Thank you!';
        form.reset();
        
        setTimeout(() => {
          submitBtn.textContent = originalText;
          submitBtn.disabled = false;
        }, 2000);
      }, 1000);
    }
  }

  // Header scroll behavior
  class HeaderBehavior {
    constructor() {
      this.header = document.querySelector('.header');
      this.lastScrollTop = 0;
      this.delta = 5;
      this.navbarHeight = this.header?.offsetHeight || 0;
      
      this.init();
    }

    init() {
      if (!this.header) return;
      
      let ticking = false;

      const updateHeader = () => {
        this.updateHeaderState();
        ticking = false;
      };

      const requestTick = () => {
        if (!ticking) {
          requestAnimationFrame(updateHeader);
          ticking = true;
        }
      };

      window.addEventListener('scroll', requestTick, { passive: true });
    }

    updateHeaderState() {
      const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

      // Add background when scrolled
      if (scrollTop > 50) {
        this.header.classList.add('is-scrolled');
      } else {
        this.header.classList.remove('is-scrolled');
      }

      this.lastScrollTop = scrollTop;
    }
  }

  // Initialize all components when DOM is ready
  function initializeApp() {
    new ScrollAnimations();
    new ParallaxEffect();
    new MobileNavigation();
    new SmoothScroll();
    new FormEnhancement();
    new HeaderBehavior();

    // Add loaded class to body for any CSS transitions
    document.body.classList.add('is-loaded');
  }

  // Load polyfills if needed and initialize
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeApp);
  } else {
    initializeApp();
  }

  // Handle page visibility changes
  document.addEventListener('visibilitychange', () => {
    if (document.visibilityState === 'visible') {
      // Re-enable animations if page becomes visible
      const elements = document.querySelectorAll('[data-animate]');
      elements.forEach(el => {
        if (el.getBoundingClientRect().top < window.innerHeight) {
          el.style.opacity = '1';
          el.style.transform = 'translateY(0) translateX(0)';
        }
      });
    }
  });

})();